<?php
/**
 * @package     AISmartTalk
 * @subpackage  plg_system_aismarttalk
 * @copyright   Copyright (C) 2024 AI SmartTalk. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;

require_once dirname(__DIR__) . '/classes/OAuthHandler.php';

class JFormFieldOauthconnect extends FormField
{
    protected $type = 'Oauthconnect';

    protected function getInput()
    {
        $app = Factory::getApplication();
        $session = Factory::getSession();
        
        // Get plugin parameters FRESH from database (important after OAuth callback)
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('params')
            ->from('#__extensions')
            ->where('element = ' . $db->quote('aismarttalk'))
            ->where('folder = ' . $db->quote('system'))
            ->where('type = ' . $db->quote('plugin'));
        $db->setQuery($query);
        $currentParams = $db->loadResult();
        $params = new Registry($currentParams);
        
        // Check for OAuth success/error messages from session
        $oauthSuccess = $session->get('aismarttalk_oauth_success', '');
        $oauthError = $session->get('aismarttalk_oauth_error', '');
        
        // Clear session messages after reading (single-use)
        if (!empty($oauthSuccess)) {
            $session->clear('aismarttalk_oauth_success');
        }
        if (!empty($oauthError)) {
            $session->clear('aismarttalk_oauth_error');
        }
        
        $oauthHandler = new OAuthHandler($params);
        $isConnected = $oauthHandler->isConnected();
        $chatModelId = $params->get('chat_model_id', '');
        $apiUrl = rtrim($params->get('api_url', 'https://aismarttalk.tech'), '/');
        
        // Build backoffice link
        $backofficeLink = '';
        if (!empty($chatModelId)) {
            $lang = Factory::getLanguage()->getTag();
            $langCode = substr($lang, 0, 2);
            $backofficeLink = $apiUrl . '/' . $langCode . '/admin/chatModel/' . $chatModelId;
        }
        
        $token = Session::getFormToken();
        $ajaxUrl = Uri::base() . 'index.php?option=com_ajax&plugin=aismarttalk&group=system&format=json&' . $token . '=1';
        
        $html = [];
        
        // Styles
        $html[] = '<style>
            .aismarttalk-oauth-container {
                padding: 20px;
                background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
                border-radius: 12px;
                margin: 10px 0;
                border: 1px solid #dee2e6;
            }
            .aismarttalk-oauth-status {
                display: flex;
                flex-direction: column;
                align-items: center;
                text-align: center;
                gap: 16px;
            }
            .aismarttalk-oauth-icon {
                width: 64px;
                height: 64px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 28px;
            }
            .aismarttalk-oauth-icon.connected {
                background: linear-gradient(135deg, #10b981 0%, #059669 100%);
                color: white;
            }
            .aismarttalk-oauth-icon.disconnected {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
            }
            .aismarttalk-oauth-title {
                font-size: 18px;
                font-weight: 600;
                color: #333;
                margin: 0;
            }
            .aismarttalk-oauth-desc {
                font-size: 14px;
                color: #666;
                margin: 0;
                max-width: 400px;
            }
            .aismarttalk-oauth-model-id {
                display: inline-block;
                background: rgba(102, 126, 234, 0.1);
                color: #667eea;
                font-size: 12px;
                font-family: monospace;
                padding: 6px 12px;
                border-radius: 6px;
                border: 1px solid rgba(102, 126, 234, 0.2);
            }
            .aismarttalk-oauth-btn-group {
                display: flex;
                gap: 12px;
                flex-wrap: wrap;
                justify-content: center;
            }
            .aismarttalk-oauth-btn {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                padding: 12px 24px;
                border: none;
                border-radius: 8px;
                cursor: pointer;
                font-size: 14px;
                font-weight: 500;
                text-decoration: none;
                transition: all 0.2s ease;
            }
            .aismarttalk-oauth-btn:disabled {
                opacity: 0.6;
                cursor: not-allowed;
            }
            .aismarttalk-oauth-btn-primary {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
            }
            .aismarttalk-oauth-btn-primary:hover:not(:disabled) {
                box-shadow: 0 4px 12px rgba(102, 126, 234, 0.4);
                transform: translateY(-1px);
            }
            .aismarttalk-oauth-btn-secondary {
                background: white;
                color: #333;
                border: 1px solid #dee2e6;
            }
            .aismarttalk-oauth-btn-secondary:hover:not(:disabled) {
                background: #f8f9fa;
                border-color: #adb5bd;
            }
            .aismarttalk-oauth-btn-danger {
                background: #dc3545;
                color: white;
            }
            .aismarttalk-oauth-btn-danger:hover:not(:disabled) {
                background: #c82333;
            }
            .aismarttalk-oauth-btn .spinner {
                display: none;
                width: 14px;
                height: 14px;
                border: 2px solid transparent;
                border-top-color: currentColor;
                border-radius: 50%;
                animation: spin 0.8s linear infinite;
            }
            .aismarttalk-oauth-btn.loading .spinner {
                display: inline-block;
            }
            .aismarttalk-oauth-btn-icon {
                width: 18px;
                height: 18px;
                border-radius: 4px;
            }
            @keyframes spin {
                to { transform: rotate(360deg); }
            }
            .aismarttalk-oauth-alert {
                margin-top: 16px;
                padding: 12px 16px;
                border-radius: 8px;
                display: none;
                width: 100%;
                max-width: 400px;
                text-align: center;
            }
            .aismarttalk-oauth-alert.success {
                display: block;
                background: #d4edda;
                color: #155724;
                border: 1px solid #c3e6cb;
            }
            .aismarttalk-oauth-alert.error {
                display: block;
                background: #f8d7da;
                color: #721c24;
                border: 1px solid #f5c6cb;
            }
        </style>';
        
        $html[] = '<div class="aismarttalk-oauth-container">';
        
        // Display OAuth success message from session (after callback redirect)
        if (!empty($oauthSuccess)) {
            $html[] = '<div class="aismarttalk-oauth-alert success" style="display:block;margin-bottom:16px;">';
            $html[] = '<strong>&#10003;</strong> ' . htmlspecialchars($oauthSuccess);
            $html[] = '</div>';
        }
        
        // Display OAuth error message from session (after callback redirect)
        if (!empty($oauthError)) {
            $html[] = '<div class="aismarttalk-oauth-alert error" style="display:block;margin-bottom:16px;">';
            $html[] = '<strong>&#9888;</strong> ' . htmlspecialchars($oauthError);
            $html[] = '</div>';
        }
        
        $html[] = '<div class="aismarttalk-oauth-status">';
        
        if ($isConnected) {
            // Connected state
            $html[] = '<div class="aismarttalk-oauth-icon connected">&#10003;</div>';
            $html[] = '<h3 class="aismarttalk-oauth-title">' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_CONNECTED') . '</h3>';
            $html[] = '<p class="aismarttalk-oauth-desc">' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_CONNECTED_DESC') . '</p>';
            $html[] = '<div class="aismarttalk-oauth-model-id">' . htmlspecialchars($chatModelId) . '</div>';
            
            $html[] = '<div class="aismarttalk-oauth-btn-group">';
            if (!empty($backofficeLink)) {
                $html[] = '<a href="' . htmlspecialchars($backofficeLink) . '" target="_blank" class="aismarttalk-oauth-btn aismarttalk-oauth-btn-secondary">';
                $html[] = '<span>&#x2197;</span>';
                $html[] = '<span>' . Text::_('PLG_SYSTEM_AISMARTTALK_OPEN_BACKOFFICE') . '</span>';
                $html[] = '</a>';
            }
            $html[] = '</div>';
            
            $html[] = '<button type="button" class="aismarttalk-oauth-btn aismarttalk-oauth-btn-danger" onclick="aismarttalkOAuthDisconnect()" id="btn-oauth-disconnect">';
            $html[] = '<span class="spinner"></span>';
            $html[] = '<span class="btn-text">' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_DISCONNECT') . '</span>';
            $html[] = '</button>';
        } else {
            // Disconnected state
            $html[] = '<div class="aismarttalk-oauth-icon disconnected">&#x2601;</div>';
            $html[] = '<h3 class="aismarttalk-oauth-title">' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_CONNECT_TITLE') . '</h3>';
            $html[] = '<p class="aismarttalk-oauth-desc">' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_CONNECT_DESC') . '</p>';
            
            $html[] = '<div class="aismarttalk-oauth-btn-group">';
            $html[] = '<button type="button" class="aismarttalk-oauth-btn aismarttalk-oauth-btn-primary" onclick="aismarttalkOAuthConnect()" id="btn-oauth-connect">';
            $html[] = '<img src="https://aismarttalk.tech/images/favicons/favicon-32.png" alt="" class="aismarttalk-oauth-btn-icon" />';
            $html[] = '<span class="spinner"></span>';
            $html[] = '<span class="btn-text">' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_CONNECT_BTN') . '</span>';
            $html[] = '</button>';
            $html[] = '</div>';
        }
        
        $html[] = '<div id="aismarttalk-oauth-alert" class="aismarttalk-oauth-alert"></div>';
        $html[] = '</div>';
        $html[] = '</div>';
        
        // JavaScript
        $html[] = '<script>
        function aismarttalkOAuthConnect() {
            var btn = document.getElementById("btn-oauth-connect");
            var alertDiv = document.getElementById("aismarttalk-oauth-alert");
            
            btn.disabled = true;
            btn.classList.add("loading");
            alertDiv.className = "aismarttalk-oauth-alert";
            alertDiv.style.display = "none";
            
            var xhr = new XMLHttpRequest();
            xhr.open("POST", "' . $ajaxUrl . '&task=oauth_connect", true);
            xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
            
            xhr.onreadystatechange = function() {
                if (xhr.readyState === 4) {
                    btn.disabled = false;
                    btn.classList.remove("loading");
                    
                    try {
                        var response = JSON.parse(xhr.responseText);
                        var data = response.data || response;
                        
                        if (Array.isArray(data) && data.length > 0) {
                            data = data[0];
                        }
                        
                        if (data && data.success && data.url) {
                            // Redirect to OAuth authorization page
                            window.location.href = data.url;
                        } else {
                            alertDiv.className = "aismarttalk-oauth-alert error";
                            alertDiv.textContent = (data && data.message) ? data.message : "' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_ERROR', true) . '";
                            alertDiv.style.display = "block";
                        }
                    } catch (e) {
                        console.error("OAuth Connect Error:", e, xhr.responseText);
                        alertDiv.className = "aismarttalk-oauth-alert error";
                        alertDiv.textContent = "' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_ERROR', true) . '";
                        alertDiv.style.display = "block";
                    }
                }
            };
            
            xhr.send();
        }
        
        function aismarttalkOAuthDisconnect() {
            if (!confirm("' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_DISCONNECT_CONFIRM', true) . '")) {
                return;
            }
            
            var btn = document.getElementById("btn-oauth-disconnect");
            var alertDiv = document.getElementById("aismarttalk-oauth-alert");
            
            btn.disabled = true;
            btn.classList.add("loading");
            alertDiv.className = "aismarttalk-oauth-alert";
            alertDiv.style.display = "none";
            
            var xhr = new XMLHttpRequest();
            xhr.open("POST", "' . $ajaxUrl . '&task=oauth_disconnect", true);
            xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
            
            xhr.onreadystatechange = function() {
                if (xhr.readyState === 4) {
                    btn.disabled = false;
                    btn.classList.remove("loading");
                    
                    try {
                        var response = JSON.parse(xhr.responseText);
                        var data = response.data || response;
                        
                        if (Array.isArray(data) && data.length > 0) {
                            data = data[0];
                        }
                        
                        if (data && data.success) {
                            alertDiv.className = "aismarttalk-oauth-alert success";
                            alertDiv.textContent = data.message || "' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_DISCONNECTED', true) . '";
                            alertDiv.style.display = "block";
                            
                            // Reload page after 1.5 seconds
                            setTimeout(function() {
                                window.location.reload();
                            }, 1500);
                        } else {
                            alertDiv.className = "aismarttalk-oauth-alert error";
                            alertDiv.textContent = (data && data.message) ? data.message : "' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_ERROR', true) . '";
                            alertDiv.style.display = "block";
                        }
                    } catch (e) {
                        console.error("OAuth Disconnect Error:", e, xhr.responseText);
                        alertDiv.className = "aismarttalk-oauth-alert error";
                        alertDiv.textContent = "' . Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_ERROR', true) . '";
                        alertDiv.style.display = "block";
                    }
                }
            };
            
            xhr.send();
        }
        </script>';
        
        return implode("\n", $html);
    }

    protected function getLabel()
    {
        return '';
    }
}
