<?php
/**
 * @package     AISmartTalk
 * @subpackage  plg_system_aismarttalk
 * @copyright   Copyright (C) 2024 AI SmartTalk. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\Registry\Registry;

require_once __DIR__ . '/EcommerceDetector.php';

/**
 * Product synchronization handler
 */
class ProductSyncHandler
{
    /**
     * Plugin parameters
     *
     * @var Registry
     */
    private $params;

    /**
     * E-commerce detector
     *
     * @var EcommerceDetector
     */
    private $detector;

    /**
     * Constructor
     *
     * @param Registry $params Plugin parameters
     */
    public function __construct($params)
    {
        $this->params = $params;
        $this->detector = new EcommerceDetector();
    }

    /**
     * Sync all products with AI SmartTalk
     *
     * @param boolean $forceSync Force re-sync of already synced products
     * @return boolean Success status
     */
    public function syncAllProducts($forceSync = false)
    {
        error_log('=== AISmartTalk ProductSync - START syncAllProducts ===');
        error_log('Force sync: ' . ($forceSync ? 'YES' : 'NO'));
        
        $component = $this->getActiveComponent();
        error_log('Active e-commerce component: ' . $component);

        if ($component === 'none') {
            error_log('ERROR: No e-commerce component detected');
            return false;
        }

        $products = $this->getProductsToSync($component, $forceSync);
        error_log('Products found: ' . count($products));

        if (empty($products)) {
            error_log('No products to sync');
            return true;
        }

        $result = $this->sendProductsToApi($products, $component);
        error_log('=== AISmartTalk ProductSync - END syncAllProducts - Result: ' . ($result ? 'SUCCESS' : 'FAILED') . ' ===');
        return $result;
    }

    /**
     * Sync specific products
     *
     * @param array $productIds Product IDs to sync
     * @param boolean $forceSync Force re-sync
     * @param string $component E-commerce component
     * @return boolean Success status
     */
    public function syncProducts($productIds, $forceSync = false, $component = null)
    {
        error_log('=== AISmartTalk ProductSync - START syncProducts ===');
        error_log('Product IDs: ' . implode(', ', $productIds));
        error_log('Force sync: ' . ($forceSync ? 'YES' : 'NO'));
        
        if ($component === null) {
            $component = $this->getActiveComponent();
        }
        error_log('Component: ' . $component);

        if ($component === 'none' || empty($productIds)) {
            error_log('ERROR: Component is none or no product IDs provided');
            return false;
        }

        $products = $this->getProductsToSync($component, $forceSync, $productIds);
        error_log('Products found: ' . count($products));

        if (empty($products)) {
            error_log('No products to sync');
            return true;
        }

        $result = $this->sendProductsToApi($products, $component);
        error_log('=== AISmartTalk ProductSync - END syncProducts - Result: ' . ($result ? 'SUCCESS' : 'FAILED') . ' ===');
        return $result;
    }

    /**
     * Get active e-commerce component
     * Always uses auto-detection
     *
     * @return string Component name
     */
    private function getActiveComponent()
    {
        return $this->detector->detect();
    }

    /**
     * Get products to synchronize (HikaShop only)
     *
     * @param string $component E-commerce component
     * @param boolean $forceSync Force re-sync
     * @param array $productIds Specific product IDs
     * @return array Products data
     */
    private function getProductsToSync($component, $forceSync = false, $productIds = [])
    {
        if ($component === 'hikashop') {
            return $this->getHikaShopProducts($forceSync, $productIds);
        }
        return [];
    }


    /**
     * Get HikaShop products
     */
    private function getHikaShopProducts($forceSync, $productIds = [])
    {
        $db = Factory::getDbo();
        
        error_log('========== HikaShop Query START ==========');
        error_log('Force sync: ' . ($forceSync ? 'YES' : 'NO'));
        error_log('Product IDs filter: ' . (!empty($productIds) ? implode(',', $productIds) : 'NONE'));
        
        // Ensure sync columns exist
        $this->ensureSyncColumnsExist('#__hikashop_product');
        
        // First, let's check if the product exists at all
        if (!empty($productIds)) {
            $checkQuery = $db->getQuery(true)
                ->select('p.product_id, p.product_name, p.product_published, p.product_type, p.product_quantity')
                ->from('#__hikashop_product AS p')
                ->where('p.product_id IN (' . implode(',', array_map('intval', $productIds)) . ')');
            $db->setQuery($checkQuery);
            $rawProduct = $db->loadAssoc();
            error_log('Raw product data: ' . json_encode($rawProduct));
        }
        
        // HikaShop stores prices in a separate table
        $query = $db->getQuery(true)
            ->select('p.product_id AS id, p.product_name AS name, p.product_description AS description')
            ->select('p.product_code AS reference, COALESCE(pr.price_value, 0) AS price, p.product_quantity AS stock')
            ->select('p.product_url AS url, p.product_published, p.product_type')
            ->from('#__hikashop_product AS p')
            ->join('LEFT', '#__hikashop_price AS pr ON p.product_id = pr.price_product_id AND pr.price_min_quantity <= 1')
            ->where('p.product_published = 1')
            ->where('p.product_type = ' . $db->quote('main'));

        if (!$forceSync) {
            $query->where('(p.aismarttalk_synch = 0 OR p.aismarttalk_synch IS NULL)');
        }

        if (!empty($productIds)) {
            $query->where('p.product_id IN (' . implode(',', array_map('intval', $productIds)) . ')');
        }

        // HikaShop: Filter by stock - unlimited stock is -1 or NULL, or quantity > 0
        $query->where('(p.product_quantity IS NULL OR p.product_quantity = -1 OR p.product_quantity > 0)');

        // Group by product_id to avoid duplicates from price table
        $query->group('p.product_id');

        error_log('Full SQL Query: ' . (string)$query);
        
        $db->setQuery($query);
        $products = $db->loadAssocList();
        
        error_log('Products found: ' . count($products));
        if (!empty($products)) {
            error_log('First product data: ' . json_encode($products[0]));
        } else {
            error_log('NO PRODUCTS FOUND - Check filters above');
        }
        error_log('========== HikaShop Query END ==========');
        
        return $products;
    }


    /**
     * Send products to AI SmartTalk API
     *
     * @param array $products Products data
     * @param string $component E-commerce component
     * @return boolean Success status
     */
    private function sendProductsToApi($products, $component)
    {
        error_log('AISmartTalk ProductSync - sendProductsToApi');
        
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');

        error_log('API URL (from config): ' . $apiUrl);
        error_log('Chat Model ID: ' . $chatModelId);
        error_log('Chat Model Token: ' . (empty($chatModelToken) ? 'EMPTY' : 'SET'));
        error_log('Products count: ' . count($products));

        // Validate API URL
        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            error_log('WARNING: Invalid API URL, using default: https://aismarttalk.tech');
            $apiUrl = 'https://aismarttalk.tech';
        }

        if (empty($chatModelId) || empty($chatModelToken)) {
            error_log('ERROR: Missing chatModelId or chatModelToken');
            return false;
        }

        $documentDatas = [];
        $syncedProductIds = [];

        foreach ($products as $product) {
            // Format product data for API
            $documentDatas[] = [
                'id' => $product['id'],
                'title' => $product['name'],
                'description' => isset($product['description']) ? strip_tags($product['description']) : '',
                'description_short' => isset($product['description_short']) ? strip_tags($product['description_short']) : '',
                'reference' => isset($product['reference']) ? $product['reference'] : '',
                'price' => isset($product['price']) ? (float)$product['price'] : 0,
                'currency' => 'EUR',
                'url' => isset($product['url']) ? $product['url'] : '',
                'type' => 'product',
            ];

            $syncedProductIds[] = $product['id'];

            // Send in batches of 10
            if (count($documentDatas) === 10) {
                if (!$this->postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)) {
                    return false;
                }
                $documentDatas = [];
            }
        }

        // Send remaining products
        if (!empty($documentDatas)) {
            if (!$this->postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)) {
                return false;
            }
        }

        // Mark products as synchronized
        if (!empty($syncedProductIds)) {
            $this->markProductsAsSynced($syncedProductIds, $component);
        }

        return true;
    }

    /**
     * Post data to API
     *
     * @param array $documentDatas Document data array
     * @param string $apiUrl API URL
     * @param string $chatModelId Chat model ID
     * @param string $chatModelToken Chat model token
     * @return boolean Success status
     */
    private function postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)
    {
        $url = rtrim($apiUrl, '/') . '/api/document/source';
        error_log('AISmartTalk ProductSync - POST to: ' . $url);
        error_log('Documents in batch: ' . count($documentDatas));

        $payload = [
            'documentDatas' => $documentDatas,
            'chatModelId' => $chatModelId,
            'chatModelToken' => $chatModelToken,
            'source' => 'JOOMLA',
        ];

        error_log('Payload: ' . json_encode($payload, JSON_PRETTY_PRINT));

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        $curlErrno = curl_errno($ch);

        error_log('HTTP Code: ' . $httpCode);
        
        if ($curlErrno) {
            error_log('CURL ERROR #' . $curlErrno . ': ' . $curlError);
        }

        if ($result === false) {
            error_log('CURL ERROR: Request failed - ' . $curlError);
            curl_close($ch);
            return false;
        }

        if ($httpCode !== 200) {
            error_log('HTTP ERROR: Expected 200, got ' . $httpCode);
            error_log('Response: ' . $result);
            curl_close($ch);
            return false;
        }

        curl_close($ch);

        error_log('Response body: ' . $result);
        
        $response = json_decode($result, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('JSON DECODE ERROR: ' . json_last_error_msg());
            return false;
        }
        
        if (isset($response['status']) && $response['status'] === 'error') {
            error_log('API ERROR: ' . (isset($response['message']) ? $response['message'] : 'Unknown error'));
            return false;
        }

        error_log('API call SUCCESS');
        return true;
    }

    /**
     * Mark products as synchronized
     *
     * @param array $productIds Product IDs to mark
     * @param string $component E-commerce component
     */
    private function markProductsAsSynced($productIds, $component)
    {
        if (empty($productIds)) {
            return;
        }

        $db = Factory::getDbo();
        $tableName = $this->getProductTableName($component);
        $idField = $this->getProductIdField($component);
        
        $query = $db->getQuery(true)
            ->update($tableName)
            ->set('aismarttalk_synch = 1')
            ->where($idField . ' IN (' . implode(',', array_map('intval', $productIds)) . ')');

        $db->setQuery($query);
        $db->execute();
    }

    /**
     * Get product table name based on component (HikaShop only)
     */
    private function getProductTableName($component)
    {
        if ($component === 'hikashop') {
            return '#__hikashop_product';
        }
        return '';
    }

    /**
     * Get product ID field name based on component (HikaShop only)
     */
    private function getProductIdField($component)
    {
        if ($component === 'hikashop') {
            return 'product_id';
        }
        return 'id';
    }

    /**
     * Ensure sync columns exist in table
     */
    private function ensureSyncColumnsExist($tableName)
    {
        $db = Factory::getDbo();
        
        try {
            $columns = $db->getTableColumns($tableName, false);
            
            if (!isset($columns['aismarttalk_synch'])) {
                $query = "ALTER TABLE " . $tableName . " ADD COLUMN `aismarttalk_synch` TINYINT(1) NOT NULL DEFAULT 0";
                $db->setQuery($query);
                $db->execute();
                error_log('AISmartTalk - Added aismarttalk_synch column to ' . $tableName);
            }
        } catch (Exception $e) {
            error_log('AISmartTalk - Error ensuring sync columns exist in ' . $tableName . ': ' . $e->getMessage());
        }
    }
}
