<?php
/**
 * @package     AISmartTalk
 * @subpackage  plg_system_aismarttalk
 * @copyright   Copyright (C) 2024 AI SmartTalk. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Session\Session;
use Joomla\Registry\Registry;

/**
 * OAuth Handler for AI SmartTalk Joomla Plugin
 * Implements OAuth 2.0 Authorization Code flow with PKCE
 */
class OAuthHandler
{
    /**
     * OAuth client ID for Joomla
     */
    const OAUTH_CLIENT_ID = 'joomla';

    /**
     * OAuth scope for Joomla integration
     */
    const OAUTH_SCOPE = 'embed chat sync:articles';

    /**
     * Plugin parameters
     *
     * @var Registry
     */
    private $params;

    /**
     * API base URL
     *
     * @var string
     */
    private $apiUrl;

    /**
     * Public API URL (for browser redirects)
     *
     * @var string
     */
    private $publicApiUrl;

    /**
     * Constructor
     *
     * @param Registry $params Plugin parameters
     */
    public function __construct(Registry $params)
    {
        $this->params = $params;
        
        // Public URL = what the browser sees (for OAuth redirects)
        // This should always be a browser-accessible URL like https://aismarttalk.tech
        $this->publicApiUrl = rtrim($params->get('api_url', 'https://aismarttalk.tech'), '/');
        
        // Internal URL = what the server uses for API calls (curl requests)
        // For Docker: set AISMARTTALK_INTERNAL_API_URL env var in docker-compose.yml
        // This allows the server to communicate with AI SmartTalk via Docker internal network
        $internalUrl = getenv('AISMARTTALK_INTERNAL_API_URL');
        if (!empty($internalUrl)) {
            $this->apiUrl = rtrim($internalUrl, '/');
        } else {
            $this->apiUrl = $this->publicApiUrl;
        }
    }

    /**
     * Get the OAuth callback URL for this Joomla installation
     *
     * @return string The callback URL
     */
    public function getCallbackUrl()
    {
        return Uri::base() . 'index.php?option=com_ajax&plugin=aismarttalk&group=system&format=json&task=oauth_callback';
    }

    /**
     * Generate PKCE code_verifier (43-128 chars, URL-safe)
     *
     * @return string The code_verifier
     */
    public function generateCodeVerifier()
    {
        return rtrim(strtr(base64_encode(random_bytes(32)), '+/', '-_'), '=');
    }

    /**
     * Generate PKCE code_challenge from code_verifier using SHA256
     *
     * @param string $codeVerifier The code_verifier
     * @return string The code_challenge
     */
    public function generateCodeChallenge($codeVerifier)
    {
        $hash = hash('sha256', $codeVerifier, true);
        return rtrim(strtr(base64_encode($hash), '+/', '-_'), '=');
    }

    /**
     * Register the OAuth redirect URI with AI SmartTalk
     * Must be called before initiating the OAuth flow
     *
     * @return array ['success' => bool, 'error' => string|null]
     */
    public function registerClient()
    {
        $callbackUrl = $this->getCallbackUrl();
        $registerUrl = $this->apiUrl . '/api/oauth/aist/clients/register';

        $payload = json_encode([
            'client_id' => self::OAUTH_CLIENT_ID,
            'redirect_uri' => $callbackUrl,
        ]);

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $registerUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Origin: ' . Uri::root(),
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($response === false) {
            return [
                'success' => false,
                'error' => $error ?: 'Connection failed',
            ];
        }

        // 200/201 = success, 409 = already registered (also success)
        if ($httpCode === 200 || $httpCode === 201 || $httpCode === 409) {
            return [
                'success' => true,
                'error' => null,
            ];
        }

        // Extract error message from response
        $errorData = json_decode($response, true);
        $errorMsg = 'HTTP ' . $httpCode;
        if (isset($errorData['error'])) {
            $errorMsg .= ': ' . $errorData['error'];
        } elseif (isset($errorData['message'])) {
            $errorMsg .= ': ' . $errorData['message'];
        }

        return [
            'success' => false,
            'error' => $errorMsg,
        ];
    }

    /**
     * Initiate the OAuth authorization flow with PKCE
     *
     * @return array ['success' => bool, 'url' => string|null, 'error' => string|null]
     */
    public function initiateFlow()
    {
        $session = Factory::getSession();

        // Step 1: Try to register the redirect URI with AI SmartTalk
        $registerResult = $this->registerClient();
        if (!$registerResult['success']) {
            error_log('AI SmartTalk: Client registration failed (non-blocking): ' . ($registerResult['error'] ?? 'Unknown error'));
        }

        // Step 2: Generate PKCE code_verifier and code_challenge
        $codeVerifier = $this->generateCodeVerifier();
        $codeChallenge = $this->generateCodeChallenge($codeVerifier);

        // Step 3: Generate state for CSRF protection
        $state = bin2hex(random_bytes(16));

        // Step 4: Store both state and code_verifier in session (10 minutes expiry handled by session)
        $session->set('aismarttalk_oauth_state', $state);
        $session->set('aismarttalk_oauth_code_verifier', $codeVerifier);
        $session->set('aismarttalk_oauth_timestamp', time());

        // Step 5: Build authorization URL with PKCE
        // Use PUBLIC URL for browser redirect (not internal Docker URL)
        $authUrl = $this->publicApiUrl . '/api/oauth/aist/authorize?' . http_build_query([
            'client_id' => self::OAUTH_CLIENT_ID,
            'redirect_uri' => $this->getCallbackUrl(),
            'response_type' => 'code',
            'scope' => self::OAUTH_SCOPE,
            'state' => $state,
            'code_challenge' => $codeChallenge,
            'code_challenge_method' => 'S256',
        ]);

        return [
            'success' => true,
            'url' => $authUrl,
            'error' => null,
        ];
    }

    /**
     * Handle the OAuth callback after user authorization (with PKCE)
     *
     * @param string $code The authorization code
     * @param string $state The state parameter for CSRF validation
     * @return array ['success' => bool, 'data' => array|null, 'error' => string|null]
     */
    public function handleCallback($code, $state)
    {
        $session = Factory::getSession();

        // Verify state for CSRF protection
        $storedState = $session->get('aismarttalk_oauth_state');
        $storedTimestamp = $session->get('aismarttalk_oauth_timestamp', 0);

        // Check if state expired (10 minutes)
        if (time() - $storedTimestamp > 600) {
            $this->clearOAuthSession();
            return [
                'success' => false,
                'data' => null,
                'error' => 'OAuth session expired. Please try again.',
            ];
        }

        if (empty($storedState) || $storedState !== $state) {
            $this->clearOAuthSession();
            return [
                'success' => false,
                'data' => null,
                'error' => 'Invalid or expired session. Please try again.',
            ];
        }

        // Get the stored code_verifier for PKCE
        $codeVerifier = $session->get('aismarttalk_oauth_code_verifier');
        if (empty($codeVerifier)) {
            $this->clearOAuthSession();
            return [
                'success' => false,
                'data' => null,
                'error' => 'PKCE verification expired. Please try again.',
            ];
        }

        // Clear the session data immediately (single-use)
        $this->clearOAuthSession();

        // Exchange code for access token using PKCE
        $tokenUrl = $this->apiUrl . '/api/oauth/aist/token';

        $payload = json_encode([
            'grant_type' => 'authorization_code',
            'code' => $code,
            'redirect_uri' => $this->getCallbackUrl(),
            'client_id' => self::OAUTH_CLIENT_ID,
            'code_verifier' => $codeVerifier,
        ]);

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $tokenUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($response === false) {
            return [
                'success' => false,
                'data' => null,
                'error' => 'Failed to exchange authorization code: ' . ($error ?: 'Connection failed'),
            ];
        }

        $body = json_decode($response, true);

        if ($httpCode !== 200 || !isset($body['access_token']) || !isset($body['chat_model_id'])) {
            $errorMessage = isset($body['error_description'])
                ? $body['error_description']
                : (isset($body['error']) ? $body['error'] : 'Unknown error');
            return [
                'success' => false,
                'data' => null,
                'error' => 'Failed to obtain access token: ' . $errorMessage,
            ];
        }

        return [
            'success' => true,
            'data' => [
                'access_token' => $body['access_token'],
                'chat_model_id' => $body['chat_model_id'],
                'scope' => $body['scope'] ?? self::OAUTH_SCOPE,
            ],
            'error' => null,
        ];
    }

    /**
     * Revoke the OAuth token
     *
     * @param string $accessToken The access token to revoke
     * @return bool True if revocation was successful
     */
    public function revokeToken($accessToken)
    {
        if (empty($accessToken)) {
            return false;
        }

        $revokeUrl = $this->apiUrl . '/api/oauth/aist/revoke';

        $payload = json_encode([
            'token' => $accessToken,
        ]);

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $revokeUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return $httpCode === 200;
    }

    /**
     * Check if the plugin is connected via OAuth
     *
     * @return bool
     */
    public function isConnected()
    {
        $connected = $this->params->get('oauth_connected', '0');
        $accessToken = $this->params->get('oauth_access_token', '');
        $chatModelId = $this->params->get('chat_model_id', '');

        return $connected === '1' && !empty($accessToken) && !empty($chatModelId);
    }

    /**
     * Clear all OAuth-related session data
     */
    private function clearOAuthSession()
    {
        $session = Factory::getSession();
        $session->clear('aismarttalk_oauth_state');
        $session->clear('aismarttalk_oauth_code_verifier');
        $session->clear('aismarttalk_oauth_timestamp');
    }

    /**
     * Save OAuth credentials to plugin parameters
     *
     * @param string $accessToken The access token
     * @param string $chatModelId The chat model ID
     * @return bool True if save was successful
     */
    public function saveCredentials($accessToken, $chatModelId)
    {
        $db = Factory::getDbo();

        try {
            // Get current plugin params
            $query = $db->getQuery(true)
                ->select('params')
                ->from('#__extensions')
                ->where('element = ' . $db->quote('aismarttalk'))
                ->where('folder = ' . $db->quote('system'))
                ->where('type = ' . $db->quote('plugin'));

            $db->setQuery($query);
            $currentParams = $db->loadResult();

            $params = new Registry($currentParams);
            $params->set('oauth_access_token', $accessToken);
            $params->set('chat_model_id', $chatModelId);
            $params->set('chat_model_token', $accessToken); // Use access_token as chat_model_token
            $params->set('oauth_connected', '1');

            // Update the plugin parameters
            $query = $db->getQuery(true)
                ->update('#__extensions')
                ->set('params = ' . $db->quote($params->toString()))
                ->where('element = ' . $db->quote('aismarttalk'))
                ->where('folder = ' . $db->quote('system'))
                ->where('type = ' . $db->quote('plugin'));

            $db->setQuery($query);
            $db->execute();

            return true;
        } catch (Exception $e) {
            error_log('AI SmartTalk: Failed to save OAuth credentials: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Clear OAuth credentials from plugin parameters
     *
     * @return bool True if clear was successful
     */
    public function clearCredentials()
    {
        $db = Factory::getDbo();

        try {
            // Get current plugin params
            $query = $db->getQuery(true)
                ->select('params')
                ->from('#__extensions')
                ->where('element = ' . $db->quote('aismarttalk'))
                ->where('folder = ' . $db->quote('system'))
                ->where('type = ' . $db->quote('plugin'));

            $db->setQuery($query);
            $currentParams = $db->loadResult();

            $params = new Registry($currentParams);
            $params->set('oauth_access_token', '');
            $params->set('chat_model_id', '');
            $params->set('chat_model_token', '');
            $params->set('oauth_connected', '0');

            // Update the plugin parameters
            $query = $db->getQuery(true)
                ->update('#__extensions')
                ->set('params = ' . $db->quote($params->toString()))
                ->where('element = ' . $db->quote('aismarttalk'))
                ->where('folder = ' . $db->quote('system'))
                ->where('type = ' . $db->quote('plugin'));

            $db->setQuery($query);
            $db->execute();

            return true;
        } catch (Exception $e) {
            error_log('AI SmartTalk: Failed to clear OAuth credentials: ' . $e->getMessage());
            return false;
        }
    }
}
