<?php
/**
 * @package     AISmartTalk
 * @subpackage  plg_system_aismarttalk
 * @copyright   Copyright (C) 2024 AI SmartTalk. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;

/**
 * E-commerce platform detector
 */
class EcommerceDetector
{
    /**
     * Detect installed e-commerce component
     * Only HikaShop is supported
     *
     * @return string Component name (hikashop, none)
     */
    public function detect()
    {
        $db = Factory::getDbo();

        // Check HikaShop (only supported e-commerce platform)
        if ($this->isComponentInstalled('com_hikashop')) {
            return 'hikashop';
        }

        return 'none';
    }

    /**
     * Check if a component is installed and enabled
     *
     * @param string $component Component name
     * @return boolean
     */
    private function isComponentInstalled($component)
    {
        $db = Factory::getDbo();

        try {
            $query = $db->getQuery(true)
                ->select('enabled')
                ->from('#__extensions')
                ->where('type = ' . $db->quote('component'))
                ->where('element = ' . $db->quote($component));

            $db->setQuery($query);
            $result = $db->loadResult();

            return $result == 1;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Get component version
     *
     * @param string $component Component name
     * @return string Version string or empty
     */
    public function getComponentVersion($component)
    {
        $db = Factory::getDbo();

        try {
            $query = $db->getQuery(true)
                ->select('manifest_cache')
                ->from('#__extensions')
                ->where('type = ' . $db->quote('component'))
                ->where('element = ' . $db->quote($component));

            $db->setQuery($query);
            $manifest = $db->loadResult();

            if ($manifest) {
                $data = json_decode($manifest, true);
                return isset($data['version']) ? $data['version'] : '';
            }
        } catch (Exception $e) {
            // Ignore errors
        }

        return '';
    }
}