<?php
/**
 * @package     AISmartTalk
 * @subpackage  plg_system_aismarttalk
 * @copyright   Copyright (C) 2024 AI SmartTalk. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\Registry\Registry;

require_once __DIR__ . '/EcommerceDetector.php';

/**
 * Clean product documents from AI SmartTalk
 */
class CleanProductDocuments
{
    /**
     * Plugin parameters
     *
     * @var Registry
     */
    private $params;

    /**
     * E-commerce detector
     *
     * @var EcommerceDetector
     */
    private $detector;

    /**
     * Constructor
     *
     * @param Registry $params Plugin parameters
     */
    public function __construct($params)
    {
        $this->params = $params;
        $this->detector = new EcommerceDetector();
    }

    /**
     * Clean all products from AI SmartTalk
     * Removes products that are deleted, unpublished, or out of stock
     */
    public function cleanAllProducts()
    {
        $component = $this->getActiveComponent();

        if ($component === 'none') {
            return false;
        }

        $activeProductIds = $this->getActiveProductIds($component);

        return $this->cleanProductsFromApi($activeProductIds, false);
    }

    /**
     * Clean specific products from AI SmartTalk
     *
     * @param array $productIds Product IDs to remove
     * @return boolean Success status
     */
    public function cleanProducts($productIds)
    {
        if (empty($productIds)) {
            return true;
        }

        return $this->cleanProductsFromApi($productIds, true);
    }

    /**
     * Get active e-commerce component
     *
     * @return string Component name
     */
    private function getActiveComponent()
    {
        $configuredComponent = $this->params->get('ecommerce_component', 'auto');

        if ($configuredComponent === 'auto') {
            return $this->detector->detect();
        }

        return $configuredComponent;
    }

    /**
     * Get active product IDs (published and in stock)
     * Only HikaShop is supported
     *
     * @param string $component E-commerce component
     * @return array Product IDs
     */
    private function getActiveProductIds($component)
    {
        if ($component === 'hikashop') {
            return $this->getHikaShopActiveProductIds();
        }
        return [];
    }

    /**
     * Get HikaShop active product IDs
     */
    private function getHikaShopActiveProductIds()
    {
        $db = Factory::getDbo();
        
        $query = $db->getQuery(true)
            ->select('product_id')
            ->from('#__hikashop_product')
            ->where('product_published = 1')
            ->where('product_type = ' . $db->quote('main'))
            ->where('product_quantity > 0');

        $db->setQuery($query);
        return $db->loadColumn();
    }

    /**
     * Clean products from AI SmartTalk API
     *
     * @param array $productIds Product IDs
     * @param boolean $deleteFromIds If true, delete these specific IDs. If false, keep only these IDs.
     * @return boolean Success status
     */
    private function cleanProductsFromApi($productIds, $deleteFromIds)
    {
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');

        // Validate API URL
        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            $apiUrl = 'https://aismarttalk.tech';
        }

        if (empty($chatModelId) || empty($chatModelToken)) {
            return false;
        }

        $url = rtrim($apiUrl, '/') . '/api/document/clean';

        $payload = [
            'productIds' => array_map('strval', $productIds),
            'chatModelId' => $chatModelId,
            'chatModelToken' => $chatModelToken,
            'deleteFromIds' => $deleteFromIds,
            'source' => 'JOOMLA',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($result === false || $httpCode !== 200) {
            curl_close($ch);
            return false;
        }

        curl_close($ch);

        $response = json_decode($result, true);
        
        if (isset($response['status']) && $response['status'] === 'error') {
            return false;
        }

        return true;
    }
}