<?php
/**
 * @package     AISmartTalk
 * @subpackage  plg_system_aismarttalk
 * @copyright   Copyright (C) 2024 AI SmartTalk. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;

/**
 * Category synchronization handler
 */
class CategorySyncHandler
{
    /**
     * Plugin parameters
     *
     * @var Registry
     */
    private $params;

    /**
     * Constructor
     *
     * @param Registry $params Plugin parameters
     */
    public function __construct($params)
    {
        $this->params = $params;
    }

    /**
     * Sync all categories with AI SmartTalk
     *
     * @param boolean $forceSync Force re-sync of already synced categories
     * @return boolean Success status
     */
    public function syncAllCategories($forceSync = false)
    {
        $categories = $this->getCategoriesToSync($forceSync);

        if (empty($categories)) {
            return true;
        }

        return $this->sendCategoriesToApi($categories);
    }

    /**
     * Sync specific categories
     *
     * @param array $categoryIds Category IDs to sync
     * @param boolean $forceSync Force re-sync
     * @return boolean Success status
     */
    public function syncCategories($categoryIds, $forceSync = false)
    {
        if (empty($categoryIds)) {
            return false;
        }

        $categories = $this->getCategoriesToSync($forceSync, $categoryIds);

        if (empty($categories)) {
            return true;
        }

        return $this->sendCategoriesToApi($categories);
    }

    /**
     * Get categories to synchronize
     *
     * @param boolean $forceSync Force re-sync
     * @param array $categoryIds Specific category IDs
     * @return array Categories data
     */
    private function getCategoriesToSync($forceSync = false, $categoryIds = [])
    {
        $db = Factory::getDbo();
        
        // Ensure sync columns exist
        $this->ensureSyncColumnsExist('#__categories');
        
        $query = $db->getQuery(true)
            ->select('c.id, c.title, c.alias, c.description, c.path, c.level')
            ->select('c.created_time, c.modified_time, c.metakey, c.metadesc')
            ->select('c.params, c.parent_id')
            ->select('p.title AS parent_title, p.alias AS parent_alias')
            ->from('#__categories AS c')
            ->join('LEFT', '#__categories AS p ON c.parent_id = p.id')
            ->where('c.published = 1')
            ->where('c.access IN (1, 5)') // Public and Registered access levels
            ->where('c.extension = ' . $db->quote('com_content'))
            ->where('c.level > 0'); // Exclude root category

        // Only sync unsynced categories if not forcing
        if (!$forceSync) {
            $query->where('(c.aismarttalk_synch = 0 OR c.aismarttalk_synch IS NULL)');
        }

        // Filter by specific category IDs if provided
        if (!empty($categoryIds)) {
            $query->where('c.id IN (' . implode(',', array_map('intval', $categoryIds)) . ')');
        }

        $query->order('c.level ASC, c.lft ASC');

        $db->setQuery($query);
        return $db->loadAssocList();
    }

    /**
     * Send categories to AI SmartTalk API
     *
     * @param array $categories Categories data
     * @return boolean Success status
     */
    private function sendCategoriesToApi($categories)
    {
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');

        // Validate API URL
        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            $apiUrl = 'https://aismarttalk.tech';
        }

        if (empty($chatModelId) || empty($chatModelToken)) {
            return false;
        }

        $documentDatas = [];
        $syncedCategoryIds = [];

        foreach ($categories as $category) {
            // Format category data for API
            $documentDatas[] = [
                'id' => 'category_' . $category['id'],
                'title' => $category['title'],
                'description' => strip_tags($category['description'] ?? ''),
                'parent_category' => $category['parent_title'] ?? '',
                'path' => $category['path'] ?? '',
                'level' => (int)$category['level'],
                'type' => 'category',
                'url' => $this->buildCategoryUrl($category),
                'keywords' => $category['metakey'] ?? '',
                'meta_description' => $category['metadesc'] ?? '',
                'created' => $category['created_time'],
                'modified' => $category['modified_time'],
            ];

            $syncedCategoryIds[] = $category['id'];

            // Send in batches of 10
            if (count($documentDatas) === 10) {
                if (!$this->postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)) {
                    return false;
                }
                $documentDatas = [];
            }
        }

        // Send remaining categories
        if (!empty($documentDatas)) {
            if (!$this->postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)) {
                return false;
            }
        }

        // Mark categories as synchronized
        if (!empty($syncedCategoryIds)) {
            $this->markCategoriesAsSynced($syncedCategoryIds);
        }

        return true;
    }

    /**
     * Build category URL
     *
     * @param array $category Category data
     * @return string Category URL
     */
    private function buildCategoryUrl($category)
    {
        // Build the URL using Joomla's routing
        $url = 'index.php?option=com_content&view=category&id=' . $category['id'] . ':' . $category['alias'];

        // Try to convert to SEF URL
        try {
            $uri = Uri::root() . ltrim($url, '/');
            return $uri;
        } catch (Exception $e) {
            // Fallback to basic URL
            return Uri::root() . 'index.php?option=com_content&view=category&id=' . $category['id'];
        }
    }

    /**
     * Post data to API
     *
     * @param array $documentDatas Document data array
     * @param string $apiUrl API URL
     * @param string $chatModelId Chat model ID
     * @param string $chatModelToken Chat model token
     * @return boolean Success status
     */
    private function postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)
    {
        $url = rtrim($apiUrl, '/') . '/api/document/source';

        $payload = [
            'documentDatas' => $documentDatas,
            'chatModelId' => $chatModelId,
            'chatModelToken' => $chatModelToken,
            'source' => 'JOOMLA',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($result === false || $httpCode !== 200) {
            curl_close($ch);
            return false;
        }

        curl_close($ch);

        $response = json_decode($result, true);
        
        if (isset($response['status']) && $response['status'] === 'error') {
            return false;
        }

        return true;
    }

    /**
     * Clean categories from AI SmartTalk
     *
     * @param array $categoryIds Category IDs to remove
     * @return boolean Success status
     */
    public function cleanCategories($categoryIds)
    {
        if (empty($categoryIds)) {
            return true;
        }

        // Prefix category IDs for the API
        $prefixedIds = array_map(function($id) {
            return 'category_' . $id;
        }, $categoryIds);

        return $this->cleanFromApi($prefixedIds, true);
    }

    /**
     * Clean all inactive categories from AI SmartTalk
     *
     * @return boolean Success status
     */
    public function cleanAllCategories()
    {
        $activeCategoryIds = $this->getActiveCategoryIds();
        
        // Prefix category IDs for the API
        $prefixedIds = array_map(function($id) {
            return 'category_' . $id;
        }, $activeCategoryIds);

        return $this->cleanFromApi($prefixedIds, false);
    }

    /**
     * Get active category IDs (published and accessible)
     *
     * @return array Category IDs
     */
    private function getActiveCategoryIds()
    {
        $db = Factory::getDbo();
        
        $query = $db->getQuery(true)
            ->select('c.id')
            ->from('#__categories AS c')
            ->where('c.published = 1')
            ->where('c.access IN (1, 5)')
            ->where('c.extension = ' . $db->quote('com_content'))
            ->where('c.level > 0');

        $db->setQuery($query);
        return $db->loadColumn();
    }

    /**
     * Clean from API
     *
     * @param array $documentIds Document IDs
     * @param boolean $deleteFromIds If true, delete these specific IDs. If false, keep only these IDs.
     * @return boolean Success status
     */
    private function cleanFromApi($documentIds, $deleteFromIds)
    {
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');

        // Validate API URL
        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            $apiUrl = 'https://aismarttalk.tech';
        }

        if (empty($chatModelId) || empty($chatModelToken)) {
            return false;
        }

        $url = rtrim($apiUrl, '/') . '/api/document/clean';

        $payload = [
            'productIds' => $documentIds,
            'chatModelId' => $chatModelId,
            'chatModelToken' => $chatModelToken,
            'deleteFromIds' => $deleteFromIds,
            'source' => 'JOOMLA',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($result === false || $httpCode !== 200) {
            curl_close($ch);
            return false;
        }

        curl_close($ch);

        $response = json_decode($result, true);
        
        if (isset($response['status']) && $response['status'] === 'error') {
            return false;
        }

        return true;
    }

    /**
     * Mark categories as synchronized
     *
     * @param array $categoryIds Category IDs to mark
     */
    private function markCategoriesAsSynced($categoryIds)
    {
        if (empty($categoryIds)) {
            return;
        }

        $db = Factory::getDbo();
        
        $query = $db->getQuery(true)
            ->update('#__categories')
            ->set('aismarttalk_synch = 1')
            ->where('id IN (' . implode(',', array_map('intval', $categoryIds)) . ')');

        $db->setQuery($query);
        $db->execute();
    }

    /**
     * Ensure sync columns exist in table
     */
    private function ensureSyncColumnsExist($tableName)
    {
        $db = Factory::getDbo();
        
        try {
            $columns = $db->getTableColumns($tableName, false);
            
            if (!isset($columns['aismarttalk_synch'])) {
                $query = "ALTER TABLE " . $tableName . " ADD COLUMN `aismarttalk_synch` TINYINT(1) NOT NULL DEFAULT 0";
                $db->setQuery($query);
                $db->execute();
                error_log('AISmartTalk - Added aismarttalk_synch column to ' . $tableName);
            }
        } catch (Exception $e) {
            error_log('AISmartTalk - Error ensuring sync columns exist in ' . $tableName . ': ' . $e->getMessage());
        }
    }
}
