<?php
/**
 * @package     AISmartTalk
 * @subpackage  plg_system_aismarttalk
 * @copyright   Copyright (C) 2024 AI SmartTalk. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;

/**
 * Article synchronization handler
 */
class ArticleSyncHandler
{
    /**
     * Plugin parameters
     *
     * @var Registry
     */
    private $params;

    /**
     * Constructor
     *
     * @param Registry $params Plugin parameters
     */
    public function __construct($params)
    {
        $this->params = $params;
    }

    /**
     * Sync all articles with AI SmartTalk
     *
     * @param boolean $forceSync Force re-sync of already synced articles
     * @return boolean Success status
     */
    public function syncAllArticles($forceSync = false)
    {
        $articles = $this->getArticlesToSync($forceSync);

        if (empty($articles)) {
            return true;
        }

        return $this->sendArticlesToApi($articles);
    }

    /**
     * Sync specific articles
     *
     * @param array $articleIds Article IDs to sync
     * @param boolean $forceSync Force re-sync
     * @return boolean Success status
     */
    public function syncArticles($articleIds, $forceSync = false)
    {
        if (empty($articleIds)) {
            return false;
        }

        $articles = $this->getArticlesToSync($forceSync, $articleIds);

        if (empty($articles)) {
            return true;
        }

        return $this->sendArticlesToApi($articles);
    }

    /**
     * Get articles to synchronize
     *
     * @param boolean $forceSync Force re-sync
     * @param array $articleIds Specific article IDs
     * @return array Articles data
     */
    private function getArticlesToSync($forceSync = false, $articleIds = [])
    {
        $db = Factory::getDbo();
        
        // Ensure sync columns exist
        $this->ensureSyncColumnsExist('#__content');
        
        $query = $db->getQuery(true)
            ->select('c.id, c.title, c.alias, c.introtext, c.fulltext, c.catid, c.featured')
            ->select('c.created, c.modified, c.metakey, c.metadesc')
            ->select('cat.title AS category_title, cat.alias AS category_alias')
            ->from('#__content AS c')
            ->join('LEFT', '#__categories AS cat ON c.catid = cat.id')
            ->where('c.state = 1')
            ->where('c.access IN (1, 5)'); // Public and Registered access levels

        // Only sync unsynced articles if not forcing
        if (!$forceSync) {
            $query->where('(c.aismarttalk_synch = 0 OR c.aismarttalk_synch IS NULL)');
        }

        // Filter by specific article IDs if provided
        if (!empty($articleIds)) {
            $query->where('c.id IN (' . implode(',', array_map('intval', $articleIds)) . ')');
        }

        $query->order('c.created DESC');

        $db->setQuery($query);
        return $db->loadAssocList();
    }

    /**
     * Send articles to AI SmartTalk API
     *
     * @param array $articles Articles data
     * @return boolean Success status
     */
    private function sendArticlesToApi($articles)
    {
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');

        // DEBUG: Log configuration
        error_log('AISmartTalk Debug - Articles Sync:');
        error_log('API URL: ' . $apiUrl);
        error_log('Chat Model ID: ' . $chatModelId);
        error_log('Chat Model Token: ' . (empty($chatModelToken) ? 'EMPTY' : 'SET'));
        error_log('Articles count: ' . count($articles));

        // Validate API URL
        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            $apiUrl = 'https://aismarttalk.tech';
        }

        if (empty($chatModelId) || empty($chatModelToken)) {
            error_log('AISmartTalk Debug - Missing configuration: chatModelId=' . $chatModelId . ', token=' . (empty($chatModelToken) ? 'empty' : 'set'));
            return false;
        }

        $documentDatas = [];
        $syncedArticleIds = [];

        foreach ($articles as $article) {
            // Combine intro and full text
            $content = trim($article['introtext']);
            if (!empty($article['fulltext'])) {
                $content .= "\n\n" . trim($article['fulltext']);
            }

            // Format article data for API
            $documentDatas[] = [
                'id' => 'article_' . $article['id'],
                'title' => $article['title'],
                'description' => strip_tags($content),
                'category' => $article['category_title'] ?? '',
                'type' => 'article',
                'url' => $this->buildArticleUrl($article),
                'keywords' => $article['metakey'] ?? '',
                'meta_description' => $article['metadesc'] ?? '',
                'featured' => (bool)$article['featured'],
                'created' => $article['created'],
                'modified' => $article['modified'],
            ];

            $syncedArticleIds[] = $article['id'];

            // Send in batches of 10
            if (count($documentDatas) === 10) {
                if (!$this->postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)) {
                    return false;
                }
                $documentDatas = [];
            }
        }

        // Send remaining articles
        if (!empty($documentDatas)) {
            if (!$this->postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)) {
                return false;
            }
        }

        // Mark articles as synchronized
        if (!empty($syncedArticleIds)) {
            $this->markArticlesAsSynced($syncedArticleIds);
        }

        return true;
    }

    /**
     * Build article URL
     *
     * @param array $article Article data
     * @return string Article URL
     */
    private function buildArticleUrl($article)
    {
        $app = Factory::getApplication();
        
        // Build the URL using Joomla's routing
        $url = 'index.php?option=com_content&view=article&id=' . $article['id'] . ':' . $article['alias'];
        
        if (!empty($article['catid'])) {
            $url .= '&catid=' . $article['catid'];
        }

        // Try to convert to SEF URL
        try {
            $uri = Uri::root() . ltrim($url, '/');
            return $uri;
        } catch (Exception $e) {
            // Fallback to basic URL
            return Uri::root() . 'index.php?option=com_content&view=article&id=' . $article['id'];
        }
    }

    /**
     * Post data to API
     *
     * @param array $documentDatas Document data array
     * @param string $apiUrl API URL
     * @param string $chatModelId Chat model ID
     * @param string $chatModelToken Chat model token
     * @return boolean Success status
     */
    private function postToApi($documentDatas, $apiUrl, $chatModelId, $chatModelToken)
    {
        $url = rtrim($apiUrl, '/') . '/api/document/source';
        error_log('AISmartTalk ArticleSync - POST to: ' . $url);
        error_log('Documents in batch: ' . count($documentDatas));

        $payload = [
            'documentDatas' => $documentDatas,
            'chatModelId' => $chatModelId,
            'chatModelToken' => $chatModelToken,
            'source' => 'JOOMLA',
        ];

        error_log('Payload: ' . json_encode($payload, JSON_PRETTY_PRINT));

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        $curlErrno = curl_errno($ch);

        error_log('HTTP Code: ' . $httpCode);
        
        if ($curlErrno) {
            error_log('CURL ERROR #' . $curlErrno . ': ' . $curlError);
        }

        if ($result === false) {
            error_log('CURL ERROR: Request failed - ' . $curlError);
            curl_close($ch);
            return false;
        }

        if ($httpCode !== 200) {
            error_log('HTTP ERROR: Expected 200, got ' . $httpCode);
            error_log('Response: ' . $result);
            curl_close($ch);
            return false;
        }

        curl_close($ch);

        error_log('Response body: ' . $result);
        
        $response = json_decode($result, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('JSON DECODE ERROR: ' . json_last_error_msg());
            return false;
        }
        
        if (isset($response['status']) && $response['status'] === 'error') {
            error_log('API ERROR: ' . (isset($response['message']) ? $response['message'] : 'Unknown error'));
            return false;
        }

        error_log('API call SUCCESS');
        return true;
    }

    /**
     * Clean articles from AI SmartTalk
     *
     * @param array $articleIds Article IDs to remove
     * @return boolean Success status
     */
    public function cleanArticles($articleIds)
    {
        if (empty($articleIds)) {
            return true;
        }

        // Prefix article IDs for the API
        $prefixedIds = array_map(function($id) {
            return 'article_' . $id;
        }, $articleIds);

        return $this->cleanFromApi($prefixedIds, true);
    }

    /**
     * Clean all inactive articles from AI SmartTalk
     *
     * @return boolean Success status
     */
    public function cleanAllArticles()
    {
        $activeArticleIds = $this->getActiveArticleIds();
        
        // Prefix article IDs for the API
        $prefixedIds = array_map(function($id) {
            return 'article_' . $id;
        }, $activeArticleIds);

        return $this->cleanFromApi($prefixedIds, false);
    }

    /**
     * Get active article IDs (published and accessible)
     *
     * @return array Article IDs
     */
    private function getActiveArticleIds()
    {
        $db = Factory::getDbo();
        
        $query = $db->getQuery(true)
            ->select('c.id')
            ->from('#__content AS c')
            ->where('c.state = 1')
            ->where('c.access IN (1, 5)');

        $db->setQuery($query);
        return $db->loadColumn();
    }

    /**
     * Clean from API
     *
     * @param array $documentIds Document IDs
     * @param boolean $deleteFromIds If true, delete these specific IDs. If false, keep only these IDs.
     * @return boolean Success status
     */
    private function cleanFromApi($documentIds, $deleteFromIds)
    {
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');

        // Validate API URL
        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            $apiUrl = 'https://aismarttalk.tech';
        }

        if (empty($chatModelId) || empty($chatModelToken)) {
            return false;
        }

        $url = rtrim($apiUrl, '/') . '/api/document/clean';

        $payload = [
            'productIds' => $documentIds,
            'chatModelId' => $chatModelId,
            'chatModelToken' => $chatModelToken,
            'deleteFromIds' => $deleteFromIds,
            'source' => 'JOOMLA',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($result === false || $httpCode !== 200) {
            curl_close($ch);
            return false;
        }

        curl_close($ch);

        $response = json_decode($result, true);
        
        if (isset($response['status']) && $response['status'] === 'error') {
            return false;
        }

        return true;
    }

    /**
     * Mark articles as synchronized
     *
     * @param array $articleIds Article IDs to mark
     */
    private function markArticlesAsSynced($articleIds)
    {
        if (empty($articleIds)) {
            return;
        }

        $db = Factory::getDbo();
        
        $query = $db->getQuery(true)
            ->update('#__content')
            ->set('aismarttalk_synch = 1')
            ->where('id IN (' . implode(',', array_map('intval', $articleIds)) . ')');

        $db->setQuery($query);
        $db->execute();
    }

    /**
     * Ensure sync columns exist in table
     */
    private function ensureSyncColumnsExist($tableName)
    {
        $db = Factory::getDbo();
        
        try {
            $columns = $db->getTableColumns($tableName, false);
            
            if (!isset($columns['aismarttalk_synch'])) {
                $query = "ALTER TABLE " . $tableName . " ADD COLUMN `aismarttalk_synch` TINYINT(1) NOT NULL DEFAULT 0";
                $db->setQuery($query);
                $db->execute();
                error_log('AISmartTalk - Added aismarttalk_synch column to ' . $tableName);
            }
        } catch (Exception $e) {
            error_log('AISmartTalk - Error ensuring sync columns exist in ' . $tableName . ': ' . $e->getMessage());
        }
    }
}
