<?php
/**
 * @package     AISmartTalk
 * @subpackage  plg_system_aismarttalk
 * @copyright   Copyright (C) 2024 AI SmartTalk. All rights reserved.
 * @license     GNU General Public License version 2 or later
 * @author      AI SmartTalk <contact@aismarttalk.tech>
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseDriver;
use Joomla\Registry\Registry;

require_once __DIR__ . '/classes/ProductSyncHandler.php';
require_once __DIR__ . '/classes/CleanProductDocuments.php';
require_once __DIR__ . '/classes/EcommerceDetector.php';
require_once __DIR__ . '/classes/ArticleSyncHandler.php';
require_once __DIR__ . '/classes/CategorySyncHandler.php';
require_once __DIR__ . '/classes/OAuthHandler.php';

/**
 * AI SmartTalk System Plugin
 */
class PlgSystemAiSmartTalk extends CMSPlugin
{
    /**
     * Application object
     *
     * @var    JApplicationCms
     */
    protected $app;

    /**
     * Database object
     *
     * @var    DatabaseDriver
     */
    protected $db;

    /**
     * Load the language file on instantiation
     *
     * @var    boolean
     */
    protected $autoloadLanguage = true;

    /**
     * AJAX handler for manual sync operations and OAuth
     * Method name must be onAjax{PluginName} for com_ajax to work
     */
    public function onAjaxAismarttalk()
    {
        $app = Factory::getApplication();
        $input = $app->input;
        $task = $input->get('task', '');

        error_log('AISmartTalk AJAX - Task: ' . $task);

        // Handle OAuth callback (can be accessed without admin check as it's a redirect from external service)
        if ($task === 'oauth_callback') {
            return $this->handleOAuthCallback();
        }

        // All other tasks require admin access
        if (!$app->isClient('administrator')) {
            return ['success' => false, 'message' => Text::_('PLG_SYSTEM_AISMARTTALK_AJAX_ACCESS_DENIED')];
        }

        // Handle OAuth tasks
        if ($task === 'oauth_connect') {
            return $this->handleOAuthConnect();
        }

        if ($task === 'oauth_disconnect') {
            return $this->handleOAuthDisconnect();
        }

        $syncType = $input->get('sync_type', 'all');

        if ($task !== 'sync') {
            return ['success' => false, 'message' => Text::_('PLG_SYSTEM_AISMARTTALK_AJAX_INVALID_TASK')];
        }

        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');

        error_log('AISmartTalk AJAX - ChatModelId: ' . $chatModelId . ', Token set: ' . (!empty($chatModelToken) ? 'YES' : 'NO'));

        if (empty($chatModelId) || empty($chatModelToken)) {
            return [
                'success' => false,
                'message' => Text::_('PLG_SYSTEM_AISMARTTALK_AJAX_MISSING_CREDENTIALS')
            ];
        }

        // Get content types from POST (current form state) or fall back to saved params
        $contentTypesFromPost = $input->get('content_types', '', 'raw');
        $contentTypesChanged = false;
        if (!empty($contentTypesFromPost)) {
            $contentTypes = json_decode($contentTypesFromPost, true);
            if (!is_array($contentTypes)) {
                $contentTypes = [];
            }
            // Check if content types changed from saved params
            $savedContentTypes = $this->params->get('content_types', []);
            if ($contentTypes !== $savedContentTypes) {
                $contentTypesChanged = true;
            }
        } else {
            $contentTypes = $this->params->get('content_types', []);
        }
        error_log('AISmartTalk AJAX - Content types: ' . json_encode($contentTypes));

        // Auto-save content types if they were sent from the form and are different
        if ($contentTypesChanged && !empty($contentTypes)) {
            $this->saveContentTypesPreference($contentTypes);
            error_log('AISmartTalk AJAX - Content types auto-saved');
        }

        if (empty($contentTypes)) {
            return [
                'success' => false,
                'message' => Text::_('PLG_SYSTEM_AISMARTTALK_AJAX_NO_CONTENT_TYPES')
            ];
        }

        try {
            $results = [];

            if ($syncType === 'all' || $syncType === 'products') {
                if (in_array('products', $contentTypes)) {
                    error_log('AISmartTalk AJAX - Starting products sync (force=true)');
                    $syncHandler = new ProductSyncHandler($this->params);
                    $results['products'] = $syncHandler->syncAllProducts(true);
                    error_log('AISmartTalk AJAX - Products sync result: ' . ($results['products'] ? 'SUCCESS' : 'FAILED'));
                }
            }

            if ($syncType === 'all' || $syncType === 'articles') {
                if (in_array('articles', $contentTypes)) {
                    error_log('AISmartTalk AJAX - Starting articles sync (force=true)');
                    $syncHandler = new ArticleSyncHandler($this->params);
                    $results['articles'] = $syncHandler->syncAllArticles(true);
                    error_log('AISmartTalk AJAX - Articles sync result: ' . ($results['articles'] ? 'SUCCESS' : 'FAILED'));
                }
            }

            if ($syncType === 'all' || $syncType === 'categories') {
                if (in_array('categories', $contentTypes)) {
                    error_log('AISmartTalk AJAX - Starting categories sync (force=true)');
                    $syncHandler = new CategorySyncHandler($this->params);
                    $results['categories'] = $syncHandler->syncAllCategories(true);
                    error_log('AISmartTalk AJAX - Categories sync result: ' . ($results['categories'] ? 'SUCCESS' : 'FAILED'));
                }
            }

            error_log('AISmartTalk AJAX - All results: ' . json_encode($results));

            if (empty($results)) {
                $typeLabel = $syncType === 'all' ? Text::_('PLG_SYSTEM_AISMARTTALK_FIELD_CONTENT_TYPES') : $syncType;
                return [
                    'success' => false,
                    'message' => Text::sprintf('PLG_SYSTEM_AISMARTTALK_AJAX_TYPE_NOT_ENABLED', ucfirst($typeLabel))
                ];
            }

            $allSuccess = !in_array(false, $results, true);
            $syncedTypes = array_keys(array_filter($results));

            if ($allSuccess) {
                $message = Text::sprintf('PLG_SYSTEM_AISMARTTALK_AJAX_SYNC_SUCCESS', implode(', ', $syncedTypes));
                if ($contentTypesChanged) {
                    $message .= ' ' . Text::_('PLG_SYSTEM_AISMARTTALK_AJAX_PREFS_SAVED');
                }
                return [
                    'success' => true,
                    'message' => $message
                ];
            } else {
                $failedTypes = array_keys(array_filter($results, function($v) { return $v === false; }));
                return [
                    'success' => false,
                    'message' => Text::sprintf('PLG_SYSTEM_AISMARTTALK_AJAX_SYNC_FAILED', implode(', ', $failedTypes))
                ];
            }
        } catch (Exception $e) {
            error_log('AISmartTalk - AJAX sync error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => Text::sprintf('PLG_SYSTEM_AISMARTTALK_AJAX_SYNC_ERROR', $e->getMessage())
            ];
        }
    }

    /**
     * Save content types preference to plugin params
     *
     * @param array $contentTypes The content types to save
     * @return bool
     */
    private function saveContentTypesPreference(array $contentTypes)
    {
        try {
            $db = Factory::getDbo();

            // Get current plugin params
            $query = $db->getQuery(true)
                ->select($db->quoteName('params'))
                ->select($db->quoteName('extension_id'))
                ->from($db->quoteName('#__extensions'))
                ->where($db->quoteName('element') . ' = ' . $db->quote('aismarttalk'))
                ->where($db->quoteName('folder') . ' = ' . $db->quote('system'))
                ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'));

            $db->setQuery($query);
            $result = $db->loadObject();

            if (!$result) {
                return false;
            }

            // Update params with new content types
            $params = new Registry($result->params);
            $params->set('content_types', $contentTypes);

            // Save back to database
            $query = $db->getQuery(true)
                ->update($db->quoteName('#__extensions'))
                ->set($db->quoteName('params') . ' = ' . $db->quote($params->toString()))
                ->where($db->quoteName('extension_id') . ' = ' . (int) $result->extension_id);

            $db->setQuery($query);
            $db->execute();

            // Update current instance params
            $this->params->set('content_types', $contentTypes);

            return true;
        } catch (Exception $e) {
            error_log('AISmartTalk - Failed to save content types: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Handle OAuth connect initiation
     *
     * @return array
     */
    private function handleOAuthConnect()
    {
        try {
            $oauthHandler = new OAuthHandler($this->params);
            $result = $oauthHandler->initiateFlow();

            if ($result['success']) {
                return [
                    'success' => true,
                    'url' => $result['url'],
                    'message' => Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_REDIRECTING')
                ];
            } else {
                return [
                    'success' => false,
                    'message' => $result['error'] ?? Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_ERROR')
                ];
            }
        } catch (Exception $e) {
            error_log('AISmartTalk - OAuth connect error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => Text::sprintf('PLG_SYSTEM_AISMARTTALK_OAUTH_ERROR_DETAIL', $e->getMessage())
            ];
        }
    }

    /**
     * Handle OAuth callback after user authorization
     *
     * @return array
     */
    private function handleOAuthCallback()
    {
        $app = Factory::getApplication();
        $input = $app->input;

        // Check for errors from OAuth provider
        $error = $input->get('error', '');
        if (!empty($error)) {
            $errorDescription = $input->get('error_description', Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_UNKNOWN_ERROR'));
            
            // Store error in session for display
            $session = Factory::getSession();
            $session->set('aismarttalk_oauth_error', $errorDescription);
            
            // Redirect back to plugin configuration (use administrator URL)
            $redirectUrl = Uri::base() . 'index.php?option=com_plugins&view=plugin&layout=edit&extension_id=' . $this->getExtensionId();
            $app->redirect($redirectUrl);
            return ['success' => false, 'message' => $errorDescription];
        }

        // Get authorization code and state
        $code = $input->get('code', '');
        $state = $input->get('state', '');

        if (empty($code) || empty($state)) {
            $session = Factory::getSession();
            $session->set('aismarttalk_oauth_error', Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_MISSING_PARAMS'));
            
            // Redirect back to plugin configuration (use administrator URL)
            $redirectUrl = Uri::base() . 'index.php?option=com_plugins&view=plugin&layout=edit&extension_id=' . $this->getExtensionId();
            $app->redirect($redirectUrl);
            return ['success' => false, 'message' => Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_MISSING_PARAMS')];
        }

        try {
            $oauthHandler = new OAuthHandler($this->params);
            $result = $oauthHandler->handleCallback($code, $state);

            if ($result['success']) {
                // Save credentials
                $saved = $oauthHandler->saveCredentials(
                    $result['data']['access_token'],
                    $result['data']['chat_model_id']
                );

                if ($saved) {
                    // Reload params after save
                    $this->loadParams();

                    // Trigger auto-sync if enabled
                    if ($this->params->get('auto_sync_enabled', 0)) {
                        $this->triggerAutoSync();
                    }

                    $session = Factory::getSession();
                    $session->set('aismarttalk_oauth_success', Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_CONNECTED_SUCCESS'));
                    
                    // Redirect back to plugin configuration (use administrator URL)
                    $redirectUrl = Uri::base() . 'index.php?option=com_plugins&view=plugin&layout=edit&extension_id=' . $this->getExtensionId();
                    $app->redirect($redirectUrl);
                    return ['success' => true, 'message' => Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_CONNECTED_SUCCESS')];
                } else {
                    $session = Factory::getSession();
                    $session->set('aismarttalk_oauth_error', Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_SAVE_FAILED'));
                    
                    // Redirect back to plugin configuration (use administrator URL)
                    $redirectUrl = Uri::base() . 'index.php?option=com_plugins&view=plugin&layout=edit&extension_id=' . $this->getExtensionId();
                    $app->redirect($redirectUrl);
                    return ['success' => false, 'message' => Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_SAVE_FAILED')];
                }
            } else {
                $session = Factory::getSession();
                $session->set('aismarttalk_oauth_error', $result['error']);
                
                // Redirect back to plugin configuration (use administrator URL)
                $redirectUrl = Uri::base() . 'index.php?option=com_plugins&view=plugin&layout=edit&extension_id=' . $this->getExtensionId();
                $app->redirect($redirectUrl);
                return ['success' => false, 'message' => $result['error']];
            }
        } catch (Exception $e) {
            $session = Factory::getSession();
            $session->set('aismarttalk_oauth_error', $e->getMessage());
            
            // Redirect back to plugin configuration (use administrator URL)
            $redirectUrl = Uri::base() . 'index.php?option=com_plugins&view=plugin&layout=edit&extension_id=' . $this->getExtensionId();
            $app->redirect($redirectUrl);
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Handle OAuth disconnect
     *
     * @return array
     */
    private function handleOAuthDisconnect()
    {
        try {
            $oauthHandler = new OAuthHandler($this->params);
            
            // Revoke token on AI SmartTalk side
            $accessToken = $this->params->get('oauth_access_token', '');
            if (!empty($accessToken)) {
                $oauthHandler->revokeToken($accessToken);
            }

            // Clear credentials
            $cleared = $oauthHandler->clearCredentials();

            if ($cleared) {
                return [
                    'success' => true,
                    'message' => Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_DISCONNECTED')
                ];
            } else {
                return [
                    'success' => false,
                    'message' => Text::_('PLG_SYSTEM_AISMARTTALK_OAUTH_DISCONNECT_FAILED')
                ];
            }
        } catch (Exception $e) {
            error_log('AISmartTalk - OAuth disconnect error: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => Text::sprintf('PLG_SYSTEM_AISMARTTALK_OAUTH_ERROR_DETAIL', $e->getMessage())
            ];
        }
    }

    /**
     * Get the extension ID for this plugin
     *
     * @return int
     */
    private function getExtensionId()
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('extension_id')
            ->from('#__extensions')
            ->where('element = ' . $db->quote('aismarttalk'))
            ->where('folder = ' . $db->quote('system'))
            ->where('type = ' . $db->quote('plugin'));
        $db->setQuery($query);
        return (int) $db->loadResult();
    }

    /**
     * Reload plugin params from database
     */
    private function loadParams()
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('params')
            ->from('#__extensions')
            ->where('element = ' . $db->quote('aismarttalk'))
            ->where('folder = ' . $db->quote('system'))
            ->where('type = ' . $db->quote('plugin'));
        $db->setQuery($query);
        $params = $db->loadResult();
        $this->params = new Registry($params);
    }

    /**
     * Trigger auto-sync for all enabled content types
     */
    private function triggerAutoSync()
    {
        $contentTypes = $this->params->get('content_types', []);

        if (in_array('products', $contentTypes)) {
            $syncHandler = new ProductSyncHandler($this->params);
            $syncHandler->syncAllProducts(true);
        }

        if (in_array('articles', $contentTypes)) {
            $syncHandler = new ArticleSyncHandler($this->params);
            $syncHandler->syncAllArticles(true);
        }

        if (in_array('categories', $contentTypes)) {
            $syncHandler = new CategorySyncHandler($this->params);
            $syncHandler->syncAllCategories(true);
        }
    }

    /**
     * Plugin installation
     */
    public function onExtensionAfterInstall($installer, $eid)
    {
        $this->addSynchFields();
    }

    /**
     * Plugin uninstallation
     */
    public function onExtensionBeforeUninstall($eid)
    {
        $this->removeSynchFields();
    }

    /**
     * Add sync fields to content tables
     */
    private function addSynchFields()
    {
        $contentTypes = $this->params->get('content_types', []);
        
        if (empty($contentTypes)) {
            $contentTypes = ['products']; // Default to products for backward compatibility
        }

        // Add sync fields for products (HikaShop only)
        if (in_array('products', $contentTypes)) {
            $ecommerceDetector = new EcommerceDetector();
            $component = $ecommerceDetector->detect();

            if ($component === 'hikashop') {
                $this->addFieldsToTable('#__hikashop_product');
            }
        }

        // Add sync fields for articles
        if (in_array('articles', $contentTypes)) {
            $this->addFieldsToTable('#__content');
        }

        // Add sync fields for categories
        if (in_array('categories', $contentTypes)) {
            $this->addFieldsToTable('#__categories');
        }
    }

    /**
     * Add fields to a specific table
     */
    private function addFieldsToTable($tableName)
    {
        $db = Factory::getDbo();
        
        try {
            // Check if aismarttalk_synch column exists
            $columns = $db->getTableColumns($tableName, false);
            
            if (!isset($columns['aismarttalk_synch'])) {
                $query = "ALTER TABLE " . $tableName . " ADD COLUMN `aismarttalk_synch` TINYINT(1) NOT NULL DEFAULT 0";
                $db->setQuery($query);
                $db->execute();
            }
        } catch (Exception $e) {
            // Silently fail if table doesn't exist
        }
    }

    /**
     * Remove sync fields from content tables
     */
    private function removeSynchFields()
    {
        // Remove from all possible tables since we don't know what was configured before
        $ecommerceDetector = new EcommerceDetector();
        $component = $ecommerceDetector->detect();

        // Remove from product tables (HikaShop only)
        if ($component === 'hikashop') {
            $this->removeFieldsFromTable('#__hikashop_product');
        }

        // Remove from articles table
        $this->removeFieldsFromTable('#__content');

        // Remove from categories table
        $this->removeFieldsFromTable('#__categories');
    }

    /**
     * Remove fields from a specific table
     */
    private function removeFieldsFromTable($tableName)
    {
        $db = Factory::getDbo();
        
        try {
            $columns = $db->getTableColumns($tableName, false);
            
            if (isset($columns['aismarttalk_synch'])) {
                $query = "ALTER TABLE " . $tableName . " DROP COLUMN `aismarttalk_synch`";
                $db->setQuery($query);
                $db->execute();
            }
        } catch (Exception $e) {
            // Silently fail if table doesn't exist
        }
    }

    /**
     * Fetch embed configuration from the API
     * 
     * @return array|null The embed configuration or null if failed
     */
    private function fetchEmbedConfig()
    {
        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        
        if (empty($chatModelId) || empty($chatModelToken) || empty($apiUrl)) {
            return null;
        }
        
        // Build the API URL for embed config
        $embedConfigUrl = rtrim($apiUrl, '/') . '/api/public/chatModel/' . urlencode($chatModelId) . '/embed-config?integrationType=JOOMLA';
        
        // Use cURL (consistent with other classes)
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $embedConfigUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $chatModelToken,
            'Content-Type: application/json',
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200 || empty($response)) {
            if (defined('JDEBUG') && JDEBUG) {
                Factory::getApplication()->enqueueMessage('AI SmartTalk: Failed to fetch embed config. HTTP Code: ' . $httpCode, 'warning');
            }
            return null;
        }
        
        $data = json_decode($response, true);
        
        if (isset($data['data'])) {
            return $data['data'];
        }
        
        return null;
    }

    /**
     * Event triggered before compiling the head
     * Injects chatbot scripts for both frontend and admin
     */
    public function onBeforeCompileHead()
    {
        // Check if chatbot is enabled
        if (!$this->params->get('enabled', 0)) {
            return;
        }

        // For admin, we'll handle it in onAfterRender
        if ($this->app->isClient('administrator')) {
            return;
        }

        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');
        $cdnUrl = $this->params->get('cdn_url', 'https://cdn.aismarttalk.tech');
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $wsUrl = $this->params->get('ws_url', 'https://ws.223.io.aismarttalk.tech');

        if (empty($chatModelId) || empty($chatModelToken)) {
            return;
        }

        // Validate URLs
        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            $apiUrl = 'https://aismarttalk.tech';
        }
        if (empty($cdnUrl) || !filter_var($cdnUrl, FILTER_VALIDATE_URL)) {
            $cdnUrl = 'https://cdn.aismarttalk.tech';
        }
        if (empty($wsUrl) || !preg_match('/^wss?:\/\//', $wsUrl)) {
            $wsUrl = 'https://ws.223.io.aismarttalk.tech';
        }

        // Get language
        $lang = Factory::getLanguage()->getTag();
        $lang = substr($lang, 0, 2); // Get only language code (en, fr, etc.)

        // Get user token from cookie if available
        $userToken = isset($_COOKIE['ai_smarttalk_oauth_token']) ? $_COOKIE['ai_smarttalk_oauth_token'] : '';

        // Fetch embed config from API
        $embedConfig = $this->fetchEmbedConfig();

        // Build chatbot settings - merge API config with local settings
        $chatbotSettings = [
            'chatModelId' => $chatModelId,
            'lang' => $lang,
            'apiUrl' => rtrim($apiUrl, '/') . '/api',
            'wsUrl' => $wsUrl,
            'cdnUrl' => $cdnUrl,
            'source' => 'JOOMLA',
        ];

        // Add user token if available
        if (!empty($userToken)) {
            $chatbotSettings['userToken'] = $userToken;
        }

        // Merge with API embed config FIRST (these are the defaults)
        if ($embedConfig && is_array($embedConfig)) {
            $protectedSettings = ['chatModelId', 'apiUrl', 'wsUrl', 'cdnUrl', 'source', 'userToken', 'lang'];

            foreach ($embedConfig as $key => $value) {
                // Only merge settings that are not protected
                if (!in_array($key, $protectedSettings)) {
                    $chatbotSettings[$key] = $value;
                }
            }
        }

        // Now apply Joomla overrides ONLY if they are explicitly set (non-empty)
        // This allows users to customize appearance while keeping API defaults when not customized
        $joomlaOverrides = [
            'buttonText' => $this->params->get('button_text', ''),
            'buttonType' => $this->params->get('button_type', ''),
            'initialColorMode' => $this->params->get('color_mode', ''),
            'chatSize' => $this->params->get('chat_size', ''),
            'position' => $this->params->get('button_position', ''),
        ];

        foreach ($joomlaOverrides as $key => $value) {
            if (!empty($value)) {
                $chatbotSettings[$key] = $value;
            }
        }

        // Boolean options - only override if explicitly set to 'on' or 'off'
        $joomlaBooleanOverrides = [
            'enableAttachment' => $this->params->get('enable_attachment', ''),
            'enableFeedback' => $this->params->get('enable_feedback', ''),
            'enableVoiceInput' => $this->params->get('enable_voice_input', ''),
            'enableVoiceMode' => $this->params->get('enable_voice_mode', ''),
        ];

        foreach ($joomlaBooleanOverrides as $key => $value) {
            if ($value === 'on' || $value === 'off') {
                $chatbotSettings[$key] = ($value === 'on');
            }
        }

        // Color overrides - only if set
        $primaryColor = $this->params->get('primary_color', '');
        $secondaryColor = $this->params->get('secondary_color', '');

        if (!empty($primaryColor) || !empty($secondaryColor)) {
            if (!isset($chatbotSettings['theme'])) {
                $chatbotSettings['theme'] = ['colors' => ['brand' => []]];
            }
            if (!isset($chatbotSettings['theme']['colors'])) {
                $chatbotSettings['theme']['colors'] = ['brand' => []];
            }
            if (!isset($chatbotSettings['theme']['colors']['brand'])) {
                $chatbotSettings['theme']['colors']['brand'] = [];
            }

            if (!empty($secondaryColor)) {
                $chatbotSettings['theme']['colors']['brand']['200'] = $secondaryColor;
            }
            if (!empty($primaryColor)) {
                $chatbotSettings['theme']['colors']['brand']['500'] = $primaryColor;
            }
        }

        // Prepare chatbot HTML (script included in HTML)
        $iframePosition = $this->params->get('iframe_position', 'footer');
        $chatbotHtml = $this->renderChatbot($chatbotSettings, $cdnUrl);

        // Store in session for later insertion
        $session = Factory::getSession();
        $session->set('aismarttalk_chatbot', $chatbotHtml);
        $session->set('aismarttalk_position', $iframePosition);
    }

    /**
     * Render chatbot using the universal embed script
     * 
     * @param array $chatbotSettings Chatbot settings array
     * @param string $cdnUrl CDN URL for the script
     * @return string The HTML/JS code to embed the chatbot
     */
    private function renderChatbot($chatbotSettings, $cdnUrl = 'https://cdn.aismarttalk.tech')
    {
        $chatbotSettingsJson = json_encode($chatbotSettings, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        // JSON should not be escaped - it's JavaScript code, not HTML attribute
        // Only escape the CDN URL for the src attribute
        return sprintf(
            "<!-- AI SmartTalk Chatbot Embedding Code -->\n<script>\nwindow.chatbotSettings = %s;\n</script>\n<script src=\"%s/universal-chatbot.js\" async></script>\n",
            $chatbotSettingsJson,
            htmlspecialchars($cdnUrl, ENT_QUOTES, 'UTF-8')
        );
    }

    /**
     * @deprecated Use renderChatbot() instead
     */
    private function renderIframe($chatModelId, $cdnUrl, $lang)
    {
        return $this->renderChatbot([], $cdnUrl);
    }

    /**
     * Event triggered after rendering the body
     * Inserts the chatbot in frontend or admin area
     */
    public function onAfterRender()
    {
        // Handle frontend chatbot insertion
        if (!$this->app->isClient('administrator')) {
            $this->insertFrontendChatbot();
            return;
        }

        // Handle admin chatbot insertion
        $this->insertAdminChatbot();
    }

    /**
     * Insert chatbot in frontend
     */
    private function insertFrontendChatbot()
    {
        $session = Factory::getSession();
        $chatbotHtml = $session->get('aismarttalk_chatbot', '');
        $iframePosition = $session->get('aismarttalk_position', 'footer');

        if (empty($chatbotHtml)) {
            return;
        }

        $body = $this->app->getBody();

        if ($iframePosition === 'before_footer') {
            // Insert before closing body tag
            $body = str_replace('</body>', $chatbotHtml . '</body>', $body);
        } else {
            // Insert in footer (before closing body tag)
            $body = str_replace('</body>', $chatbotHtml . '</body>', $body);
        }

        $this->app->setBody($body);
    }

    /**
     * Insert chatbot in admin area
     */
    private function insertAdminChatbot()
    {
        // Only display on plugin configuration page
        $input = Factory::getApplication()->input;
        $option = $input->get('option');
        $view = $input->get('view');

        if ($option !== 'com_plugins' || $view !== 'plugin') {
            return;
        }

        // Check if chatbot is enabled
        if (!$this->params->get('enabled', 0)) {
            return;
        }

        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');
        $cdnUrl = $this->params->get('cdn_url', 'https://cdn.aismarttalk.tech');
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $wsUrl = $this->params->get('ws_url', 'wss://ws.223.io.aismarttalk.tech');

        if (empty($chatModelId) || empty($chatModelToken)) {
            return;
        }

        // Validate URLs
        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            $apiUrl = 'https://aismarttalk.tech';
        }
        if (empty($cdnUrl) || !filter_var($cdnUrl, FILTER_VALIDATE_URL)) {
            $cdnUrl = 'https://cdn.aismarttalk.tech';
        }
        if (empty($wsUrl) || !preg_match('/^wss?:\/\//', $wsUrl)) {
            $wsUrl = 'wss://ws.223.io.aismarttalk.tech';
        }

        // Get language
        $lang = Factory::getLanguage()->getTag();
        $lang = substr($lang, 0, 2);

        // Get user token from cookie if available
        $userToken = isset($_COOKIE['ai_smarttalk_oauth_token']) ? $_COOKIE['ai_smarttalk_oauth_token'] : '';

        // Fetch embed config from API
        $embedConfig = $this->fetchEmbedConfig();

        // Build chatbot settings - merge API config with local settings
        $chatbotSettings = [
            'chatModelId' => $chatModelId,
            'lang' => $lang,
            'apiUrl' => rtrim($apiUrl, '/') . '/api',
            'wsUrl' => $wsUrl,
            'cdnUrl' => $cdnUrl,
            'source' => 'JOOMLA',
        ];

        // Add user token if available
        if (!empty($userToken)) {
            $chatbotSettings['userToken'] = $userToken;
        }

        // Merge with API embed config FIRST (these are the defaults)
        if ($embedConfig && is_array($embedConfig)) {
            $protectedSettings = ['chatModelId', 'apiUrl', 'wsUrl', 'cdnUrl', 'source', 'userToken', 'lang'];

            foreach ($embedConfig as $key => $value) {
                // Only merge settings that are not protected
                if (!in_array($key, $protectedSettings)) {
                    $chatbotSettings[$key] = $value;
                }
            }
        }

        // Now apply Joomla overrides ONLY if they are explicitly set (non-empty)
        $joomlaOverrides = [
            'buttonText' => $this->params->get('button_text', ''),
            'buttonType' => $this->params->get('button_type', ''),
            'initialColorMode' => $this->params->get('color_mode', ''),
            'chatSize' => $this->params->get('chat_size', ''),
            'position' => $this->params->get('button_position', ''),
        ];

        foreach ($joomlaOverrides as $key => $value) {
            if (!empty($value)) {
                $chatbotSettings[$key] = $value;
            }
        }

        // Boolean options - only override if explicitly set to 'on' or 'off'
        $joomlaBooleanOverrides = [
            'enableAttachment' => $this->params->get('enable_attachment', ''),
            'enableFeedback' => $this->params->get('enable_feedback', ''),
            'enableVoiceInput' => $this->params->get('enable_voice_input', ''),
            'enableVoiceMode' => $this->params->get('enable_voice_mode', ''),
        ];

        foreach ($joomlaBooleanOverrides as $key => $value) {
            if ($value === 'on' || $value === 'off') {
                $chatbotSettings[$key] = ($value === 'on');
            }
        }

        // Color overrides - only if set
        $primaryColor = $this->params->get('primary_color', '');
        $secondaryColor = $this->params->get('secondary_color', '');

        if (!empty($primaryColor) || !empty($secondaryColor)) {
            if (!isset($chatbotSettings['theme'])) {
                $chatbotSettings['theme'] = ['colors' => ['brand' => []]];
            }
            if (!isset($chatbotSettings['theme']['colors'])) {
                $chatbotSettings['theme']['colors'] = ['brand' => []];
            }
            if (!isset($chatbotSettings['theme']['colors']['brand'])) {
                $chatbotSettings['theme']['colors']['brand'] = [];
            }

            if (!empty($secondaryColor)) {
                $chatbotSettings['theme']['colors']['brand']['200'] = $secondaryColor;
            }
            if (!empty($primaryColor)) {
                $chatbotSettings['theme']['colors']['brand']['500'] = $primaryColor;
            }
        }

        // Render chatbot HTML (script included in HTML)
        $chatbotHtml = $this->renderChatbot($chatbotSettings, $cdnUrl);

        // Insert chatbot in body
        $body = $this->app->getBody();
        $body = str_replace('</body>', $chatbotHtml . '</body>', $body);
        $this->app->setBody($body);
    }

    /**
     * Event triggered after routing
     * Handles custom admin pages
     */
    public function onExtensionAfterSave($context, $table, $isNew)
    {
        if ($context !== 'com_plugins.plugin') {
            return;
        }

        if ($table->element !== 'aismarttalk' || $table->folder !== 'system') {
            return;
        }

        $newParams = new Registry($table->params);

        $oldChatModelId = $this->params->get('chat_model_id', '');
        $oldChatModelToken = $this->params->get('chat_model_token', '');
        $newChatModelId = $newParams->get('chat_model_id', '');
        $newChatModelToken = $newParams->get('chat_model_token', '');

        $chatModelChanged = ($oldChatModelId !== $newChatModelId) || ($oldChatModelToken !== $newChatModelToken);

        if ($chatModelChanged && !empty($newChatModelId) && !empty($newChatModelToken)) {
            error_log('AISmartTalk - Chat model changed, forcing full resync');
            $this->resetAllSyncFlags();
            $this->forceFullResync($newParams);
        } else if ($newParams->get('auto_sync_enabled', 0)) {
            $this->syncUnsyncedContentWithParams($newParams);
        }
    }

    /**
     * Reset all sync flags to trigger full resync
     */
    private function resetAllSyncFlags()
    {
        $db = Factory::getDbo();
        $contentTypes = $this->params->get('content_types', []);

        if (in_array('products', $contentTypes)) {
            $ecommerceDetector = new EcommerceDetector();
            if ($ecommerceDetector->detect() === 'hikashop') {
                try {
                    $columns = $db->getTableColumns('#__hikashop_product', false);
                    if (isset($columns['aismarttalk_synch'])) {
                        $query = $db->getQuery(true)
                            ->update('#__hikashop_product')
                            ->set('aismarttalk_synch = 0');
                        $db->setQuery($query);
                        $db->execute();
                    }
                } catch (Exception $e) {
                    error_log('AISmartTalk - Error resetting product sync flags: ' . $e->getMessage());
                }
            }
        }

        if (in_array('articles', $contentTypes)) {
            try {
                $columns = $db->getTableColumns('#__content', false);
                if (isset($columns['aismarttalk_synch'])) {
                    $query = $db->getQuery(true)
                        ->update('#__content')
                        ->set('aismarttalk_synch = 0');
                    $db->setQuery($query);
                    $db->execute();
                }
            } catch (Exception $e) {
                error_log('AISmartTalk - Error resetting article sync flags: ' . $e->getMessage());
            }
        }

        if (in_array('categories', $contentTypes)) {
            try {
                $columns = $db->getTableColumns('#__categories', false);
                if (isset($columns['aismarttalk_synch'])) {
                    $query = $db->getQuery(true)
                        ->update('#__categories')
                        ->set('aismarttalk_synch = 0');
                    $db->setQuery($query);
                    $db->execute();
                }
            } catch (Exception $e) {
                error_log('AISmartTalk - Error resetting category sync flags: ' . $e->getMessage());
            }
        }

        error_log('AISmartTalk - All sync flags reset');
    }

    /**
     * Force full resync with new params
     */
    private function forceFullResync($params)
    {
        if (!$params->get('auto_sync_enabled', 0)) {
            return;
        }

        $contentTypes = $params->get('content_types', []);

        if (in_array('products', $contentTypes)) {
            $syncHandler = new ProductSyncHandler($params);
            $syncHandler->syncAllProducts(true);
        }

        if (in_array('articles', $contentTypes)) {
            $syncHandler = new ArticleSyncHandler($params);
            $syncHandler->syncAllArticles(true);
        }

        if (in_array('categories', $contentTypes)) {
            $syncHandler = new CategorySyncHandler($params);
            $syncHandler->syncAllCategories(true);
        }
    }

    /**
     * Sync unsynced content with specific params
     */
    private function syncUnsyncedContentWithParams($params)
    {
        if (!$params->get('auto_sync_enabled', 0)) {
            return;
        }

        $contentTypes = $params->get('content_types', []);

        if (in_array('products', $contentTypes)) {
            $syncHandler = new ProductSyncHandler($params);
            $syncHandler->syncAllProducts(false);
        }

        if (in_array('articles', $contentTypes)) {
            $syncHandler = new ArticleSyncHandler($params);
            $syncHandler->syncAllArticles(false);
        }

        if (in_array('categories', $contentTypes)) {
            $syncHandler = new CategorySyncHandler($params);
            $syncHandler->syncAllCategories(false);
        }
    }

    /**
     * HikaShop: After product update
     */
    public function onAfterProductUpdate(&$product)
    {
        if (!$this->params->get('auto_sync_enabled', 0) || !$this->isContentTypeEnabled('products')) {
            return;
        }

        $ecommerceDetector = new EcommerceDetector();
        if ($ecommerceDetector->detect() !== 'hikashop') {
            return;
        }

        $productId = isset($product->product_id) ? $product->product_id : null;
        
        if ($productId) {
            $isPublished = isset($product->product_published) && $product->product_published == 1;
            $quantity = isset($product->product_quantity) ? $product->product_quantity : null;
            
            // HikaShop represents unlimited stock as -1 or NULL
            $hasStock = false;
            if ($quantity === null || $quantity === '' || $quantity == -1) {
                // Unlimited stock
                $hasStock = true;
            } elseif (is_numeric($quantity) && (int)$quantity > 0) {
                // Has stock
                $hasStock = true;
            }
            
            // Product must be published AND have stock to be synced
            if ($isPublished && $hasStock) {
                // Product is published and in stock - sync it
                $this->syncProductWithCooldown($productId, 'hikashop');
            } else {
                // Product is unpublished or out of stock - remove from AI SmartTalk
                $this->cleanProduct($productId, 'hikashop');
            }
        }
    }


    /**
     * Sync product immediately
     */
    private function syncProductWithCooldown($productId, $component)
    {
        error_log('=== AISmartTalk Plugin - syncProduct ===');
        error_log('Product ID: ' . $productId);
        error_log('Component: ' . $component);
        
        // Perform sync (force=true to always update)
        error_log('Starting product sync...');
        $syncHandler = new ProductSyncHandler($this->params);
        $syncHandler->syncProducts([$productId], true, $component);
        error_log('Product sync completed');
    }

    /**
     * Ensure sync columns exist in table
     */
    private function ensureSyncColumnsExist($tableName)
    {
        $db = Factory::getDbo();
        
        try {
            $columns = $db->getTableColumns($tableName, false);
            
            if (!isset($columns['aismarttalk_synch'])) {
                $query = "ALTER TABLE " . $tableName . " ADD COLUMN `aismarttalk_synch` TINYINT(1) NOT NULL DEFAULT 0";
                $db->setQuery($query);
                $db->execute();
                error_log('AISmartTalk - Added aismarttalk_synch column to ' . $tableName);
            }
        } catch (Exception $e) {
            error_log('AISmartTalk - Error ensuring sync columns exist: ' . $e->getMessage());
        }
    }

    /**
     * Get product table name based on component (HikaShop only)
     */
    private function getProductTableName($component)
    {
        if ($component === 'hikashop') {
            return '#__hikashop_product';
        }
        return '';
    }

    /**
     * Get product ID field name based on component (HikaShop only)
     */
    private function getProductIdField($component)
    {
        if ($component === 'hikashop') {
            return 'product_id';
        }
        return 'id';
    }

    /**
     * Clean a single product from AI SmartTalk
     */
    private function cleanProduct($productId, $component)
    {
        $apiUrl = $this->params->get('api_url', 'https://aismarttalk.tech');
        $chatModelId = $this->params->get('chat_model_id', '');
        $chatModelToken = $this->params->get('chat_model_token', '');

        if (empty($apiUrl) || !filter_var($apiUrl, FILTER_VALIDATE_URL)) {
            $apiUrl = 'https://aismarttalk.tech';
        }

        if (empty($chatModelId) || empty($chatModelToken)) {
            return false;
        }

        $url = rtrim($apiUrl, '/') . '/api/document/clean';

        $payload = [
            'productIds' => [(string)$productId],
            'chatModelId' => $chatModelId,
            'chatModelToken' => $chatModelToken,
            'deleteFromIds' => true,  // Delete this specific product
            'source' => 'JOOMLA',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        curl_close($ch);

        return ($result !== false && $httpCode === 200);
    }


    /**
     * Check if a content type is enabled for sync
     *
     * @param string $contentType Content type (products, articles, categories)
     * @return boolean
     */
    private function isContentTypeEnabled($contentType)
    {
        $contentTypes = $this->params->get('content_types', []);
        return in_array($contentType, $contentTypes);
    }

    /**
     * Event: After content save (Joomla articles)
     */
    public function onContentAfterSave($context, $article, $isNew)
    {
        error_log('AISmartTalk Debug - onContentAfterSave triggered');
        error_log('Auto sync enabled: ' . ($this->params->get('auto_sync_enabled', 0) ? 'YES' : 'NO'));
        error_log('Articles enabled: ' . ($this->isContentTypeEnabled('articles') ? 'YES' : 'NO'));
        error_log('Context: ' . $context);
        error_log('Article ID: ' . $article->id);
        error_log('Article state: ' . $article->state);
        error_log('Article access: ' . $article->access);

        if (!$this->params->get('auto_sync_enabled', 0) || !$this->isContentTypeEnabled('articles')) {
            error_log('AISmartTalk Debug - Auto sync disabled or articles not enabled');
            return;
        }

        if ($context !== 'com_content.article') {
            error_log('AISmartTalk Debug - Wrong context: ' . $context);
            return;
        }

        // Check if article is published and accessible
        if ($article->state != 1 || !in_array($article->access, [1, 5])) {
            error_log('AISmartTalk Debug - Article not published or not accessible');
            return;
        }

        error_log('AISmartTalk Debug - Starting sync for article ID: ' . $article->id);
        // Sync directly - no cooldown, no tracking
        $syncHandler = new ArticleSyncHandler($this->params);
        $result = $syncHandler->syncArticles([$article->id], true);
        error_log('AISmartTalk Debug - Sync result: ' . ($result ? 'SUCCESS' : 'FAILED'));
    }

    /**
     * Event: After content change state (Joomla articles)
     */
    public function onContentChangeState($context, $pks, $value)
    {
        if (!$this->params->get('auto_sync_enabled', 0) || !$this->isContentTypeEnabled('articles')) {
            return;
        }

        if ($context !== 'com_content.article') {
            return;
        }

        $syncHandler = new ArticleSyncHandler($this->params);
 
        foreach ($pks as $pk) {
            if ($value == 1) {
                // Article published - sync it
                $syncHandler->syncArticles([$pk], true);
            } else {
                // Article unpublished - remove from AI SmartTalk
                $syncHandler->cleanArticles([$pk]);
            }
        }
    }

    /**
     * Event: After category save
     */
    public function onCategoryAfterSave($context, $category, $isNew)
    {
        if (!$this->params->get('auto_sync_enabled', 0) || !$this->isContentTypeEnabled('categories')) {
            return;
        }

        if ($context !== 'com_categories.category' || $category->extension !== 'com_content') {
            return;
        }

        // Check if category is published and accessible
        if ($category->published != 1 || !in_array($category->access, [1, 5])) {
            return;
        }

        // Sync directly - no cooldown, no tracking
        $syncHandler = new CategorySyncHandler($this->params);
        $syncHandler->syncCategories([$category->id], true);
    }
}

